<?php
/**
 * @author Host.it
 * @package FASTCACHE::plugins::system
 * @Copyright (C) 2026 - Host.it
 * @license GNU/GPLv2 http://www.gnu.org/licenses/gpl-2.0.html
 */
// No direct access
defined ( '_JEXEC' ) or die ();

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Event\SubscriberInterface;
use Joomla\Event\Event;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Language\Language;
use Joomla\CMS\Language\LanguageHelper;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\Pluginhelper;
use Joomla\CMS\HTML\HTMLHelper;
use Fastcache\Uri as FastcacheUri;
use Fastcache\Cache as FastcacheCache;
use Fastcache\Helper as FastcacheHelper;
use Fastcache\JsonManager as FastcacheJsonManager;

if (! defined ( 'FASTCACHE_VERSION' )) {
	$currentVersion = strval ( simplexml_load_file ( JPATH_ROOT . '/plugins/system/fastcache/fastcache.xml' )->version );
	define ( 'FASTCACHE_VERSION', $currentVersion );
}

include_once (dirname ( __FILE__ ) . '/Framework/loader.php');
class PlgSystemFastcache extends CMSPlugin implements SubscriberInterface {
	/**
	 * App reference
	 *
	 * @access protected
	 * @var Object
	 */
	protected $appInstance;
	
	/**
	 * Clear the Fastcache cache system
	 *
	 * @return boolean
	 */
	private function clearCache() {
		// Clear all caches, plugin and page cache, additionally trigger plugin and HTTP headers for cache cleaning
		$outputCache = FastcacheCache::getCacheObject ();
		$staticCache = FastcacheCache::getCacheObject ( 'targetcache' );
		$pageCache = Factory::getContainer()->get(\Joomla\CMS\Cache\CacheControllerFactoryInterface::class)->createCacheController( 'output', array() );
		
		if($outputCache->clean ( 'plg_fastcache' ) && $outputCache->clean ( 'plg_fastcache_nowebp' )) {
			$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_PLUGIN_CACHE_SUCCESSFULLY_CLEARED'));
		}
		
		// Check if also images should be deleted
		$preserveImagesCache = $this->params->get ( 'preserve_cached_images', 0 );
		if($preserveImagesCache) {
			if($staticCache->clean ('js') && $staticCache->clean ('css')) {
				$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_STATIC_CACHE_SUCCESSFULLY_CLEARED'));
			}
		} else {
			if($staticCache->clean ()) {
				$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_STATIC_CACHE_SUCCESSFULLY_CLEARED'));
			}
		}
		
		
		if($pageCache->clean ( 'page' )) {
			$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_PAGE_CACHE_SUCCESSFULLY_CLEARED'));
		}
		
		// Clear HTML page cache files
		if ($this->params->get('htaccess_cache_enable', '0')) {
			$pageCacheDir = JPATH_ROOT . '/cache/fastcache/page';
			if (is_dir($pageCacheDir)) {
				$files = glob($pageCacheDir . '/*.html');
				$cleared = 0;
				foreach ($files as $file) {
					if (is_file($file)) {
						@unlink($file);
						$cleared++;
					}
				}
				if ($cleared > 0) {
					$this->appInstance->enqueueMessage(Text::sprintf('PLG_FASTCACHE_HTML_PAGE_CACHE_CLEARED', $cleared));
				}
			}
		}
		
		if(PluginHelper::getPlugin('system', 'pagecacheextended')) {
			if($pageCache->clean ( 'pce' )) {
				$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_PAGE_CACHE_PCE_SUCCESSFULLY_CLEARED'));
			}
			
			if($pageCache->clean ( 'pce-gzip' )) {
				$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_PAGE_CACHE_PCE_GZIP_SUCCESSFULLY_CLEARED'));
			}
		}
		
		// Trigger LiteSpeed cache clearing
		$this->appInstance->getDispatcher()->dispatch('onLSCacheExpired', new Event('onLSCacheExpired', []));
		
		$this->appInstance->setHeader( 'X-LiteSpeed-Purge', '*' );
		
		if($this->params->get('clear_server_cache', 0)) {
			FastcacheCache::purgeServerCache(Uri::root(false));
			$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_SERVER_CACHE_SUCCESSFULLY_CLEARED'));
		}
		
		// Purge della cache server-side con token (stile WordPress)
		// Viene eseguito SOLO quando si clicca "Clear Cache" in backend
		if($this->params->get('enable_server_cache', 0)) {
			$token = trim($this->params->get('server_cache_token', ''));
			
			// Solo se il token è impostato
			if (!empty($token)) {
				try {
					$host = rtrim(Uri::root(false), '/');
					
					$purgeSuccess = $this->execServerCachePurge($host, $token, 'regexp', '/.*');
					
					if ($purgeSuccess) {
						$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_SERVER_SIDE_CACHE_PURGED'));
					} else {
						$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_SERVER_SIDE_CACHE_PURGE_ERROR'), 'warning');
					}
				} catch (\Exception $e) {
					$this->appInstance->enqueueMessage(Text::sprintf('PLG_FASTCACHE_SERVER_SIDE_CACHE_PURGE_EXCEPTION', $e->getMessage()), 'error');
				}
			}
		}

		return true;
	}
	
	/**
	 * Verifica se l'URL corrente è in lista di esclusione per cache server
	 * Usato per decidere se inviare header cache o meno
	 *
	 * @param   string  $excludedUrls  Lista URL esclusi (uno per riga dal parametro)
	 * @return  int     1 se escluso, 0 se non escluso
	 */
	private function isServerCacheUrlExcluded($excludedUrls) {
		$uri = Uri::getInstance();
		
		$currentUrl  = $uri->toString();
		$currentPath = $uri->getPath();
		$currentQS   = (string) $uri->getQuery();
		
		$u = strtolower($currentUrl);
		$p = strtolower($currentPath);
		$q = strtolower($currentQS);
		
		// SAFETY: se non è HTML, consideralo escluso (utile anche se qualcuno chiamasse qui senza il check in addCacheHeaders)
		$document = $this->appInstance->getDocument();
		if (!$document || $document->getType() !== 'html') {
			return 1;
		}
		
		/**
		 * 0) ESCLUSIONI SITEMAP (default)
		 */
		$defaultSitemapPatterns = [
				'/sitemap',            // /sitemap.xml, /sitemap_index.xml, /sitemap.xml.gz ecc.
				'sitemap.xml',
				'sitemap_index.xml',
				'.xml',                // extra safety
				'.xml.gz',
				'format=xml',
				'option=com_jsitemap', // JSitemap
				'option=com_osmap',    // OSMap
				'view=sitemap',
		];
		
		foreach ($defaultSitemapPatterns as $needle) {
			$needle = strtolower($needle);
			if ($needle !== '' && (strpos($u, $needle) !== false || strpos($p, $needle) !== false || strpos($q, $needle) !== false)) {
				return 1;
			}
		}
		
		/**
		 * 1) ESCLUSIONI DEFAULT (path-based) - e-commerce/dinamiche
		 */
		$defaultPathPatterns = [
				'/cart', '/carrello', '/basket',
				'/checkout', '/cassa',
				'/order', '/ordine', '/ordini', '/orders',
				'/payment', '/pagamento',
				'/confirm', '/conferma', '/thank-you', '/thankyou', '/success',
				'/shipping', '/spedizione',
				'/login', '/logout', '/registrazione', '/register', '/account', '/my-account',
				'/profile', '/profilo', '/user', '/users',
				'/wishlist', '/wish-list', '/compare', '/comparazione',
				'/coupon', '/buono', '/voucher', '/giftcard', '/gift-card',
		];
		
		foreach ($defaultPathPatterns as $needle) {
			$needle = strtolower($needle);
			if ($needle !== '' && (strpos($p, $needle) !== false || strpos($u, $needle) !== false)) {
				return 1;
			}
		}
		
		/**
		 * 2) ESCLUSIONI DEFAULT (query-based)
		 */
		$defaultQueryNeedles = [
				// VirtueMart
				'option=com_virtuemart',
				'view=cart', 'view=checkout', 'view=user', 'view=orders', 'view=order',
				
				// HikaShop
				'option=com_hikashop',
				'ctrl=checkout', 'ctrl=cart', 'ctrl=user', 'ctrl=order',
				'task=checkout', 'task=cart', 'task=login', 'task=register',
				
				// J2Store
				'option=com_j2store',
				'view=carts', 'view=cart', 'view=checkout', 'view=myprofile', 'view=orders', 'view=order',
				
				// RedSHOP
				'option=com_redshop',
				'view=cart', 'view=checkout', 'view=order', 'view=account',
				
				// Joomla users core (azioni dinamiche)
				'option=com_users',
				'view=login', 'view=registration', 'view=profile', 'task=user.login', 'task=user.logout',
				
				// com_ajax: NON cachare mai
				'option=com_ajax',
		];
		
		foreach ($defaultQueryNeedles as $needle) {
			$needle = strtolower($needle);
			if ($needle !== '' && (strpos($q, $needle) !== false || strpos($u, $needle) !== false)) {
				return 1;
			}
		}
		
		/**
		 * 3) ESCLUSIONI CUSTOM (parametro admin)
		 */
		if (!empty($excludedUrls)) {
			$urlsArray = preg_split("/\r\n|\n|\r/", (string)$excludedUrls);
			
			foreach ($urlsArray as $excludedUrl) {
				$excludedUrl = trim($excludedUrl);
				if ($excludedUrl === '') {
					continue;
				}
				
				$excludedUrlLow = strtolower($excludedUrl);
				
				// wildcard finale
				if (substr($excludedUrlLow, -1) === '*') {
					$pattern = rtrim($excludedUrlLow, '*');
					if ($pattern !== '' && (strpos($u, $pattern) !== false || strpos($p, $pattern) !== false || strpos($q, $pattern) !== false)) {
						return 1;
					}
				} else {
					if (strpos($u, $excludedUrlLow) !== false || strpos($p, $excludedUrlLow) !== false || strpos($q, $excludedUrlLow) !== false) {
						return 1;
					}
				}
			}
		}
		
		return 0;
	}
	
	/**
	 * Execute cache purge on server-side cache (Varnish/CDN)
	 * Stile WordPress: endpoint + path e header method exact/regexp
	 *
	 * @param   string  $host        Root del sito (es: https://example.com o https://example.com/subdir)
	 * @param   string  $token       Security token
	 * @param   string  $method      "exact" o "regexp"
	 * @param   string  $path        Path da purgare (es: "/chi-siamo/" oppure "/.*" oppure "/")
	 *
	 * @return  boolean  True on success (200 or 404)
	 */
	private function execServerCachePurge($host, $token, $method = 'regexp', $path = '/.*') {
		$parsedUrl = parse_url($host);
		$hostname  = isset($parsedUrl['host']) ? $parsedUrl['host'] : '';

		if (empty($hostname)) {
			return false;
		}

		// Normalizza method
		$method = strtolower(trim((string)$method));
		if ($method !== 'exact' && $method !== 'regexp') {
			$method = 'regexp';
		}

		// Normalizza path
		$path = trim((string)$path);
		if ($path === '') {
			$path = '/';
		}
		// In WP: homepage -> endpoint base (path vuoto). Qui manteniamo "/" coerente
		if ($path[0] !== '/') {
			$path = '/' . $path;
		}

		// Se exact: evitiamo roba tipo "/.*"
		if ($method === 'exact' && $path === '/.*') {
			$path = '/';
		}

		// Endpoint base purge (equivalente a FASTCACHEHOST_HOST_ENDPOINTCACHE in WP)
		// NB: qui l'endpoint � sempre quello hostcdn.
		$endpointBase = 'https://purge.hostcdn.it:8443/';

		// Componi URL finale come WP: endpoint + ltrim(path,'/')
		// - regexp all: endpoint + ".*"
		// - exact page: endpoint + "chi-siamo/"
		$finalUrl = $endpointBase . ltrim($path, '/');

		$curl = curl_init();
		curl_setopt($curl, CURLOPT_URL, $finalUrl);
		curl_setopt($curl, CURLOPT_CUSTOMREQUEST, "PURGE");
		curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($curl, CURLOPT_HTTPHEADER, [
			'host: ' . $hostname,
			'X-HST-CACHE-PurgeKey: ' . $token,
			'X-HST-CACHE-Purge-Method: ' . $method
		]);
		curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);
		curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
		curl_setopt($curl, CURLOPT_TIMEOUT, 10);

		$response = curl_exec($curl);
		$httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		curl_close($curl);

		// Come WordPress: 200 e 404 sono successo
		return ($httpCode == 200 || $httpCode == 404);
	}

	/**
	 * Gets the name of the current Editor
	 *
	 * @staticvar string $sEditor
	 * @return string
	 */
	protected function excludeEditorViews() {
		$aEditors = method_exists('Joomla\CMS\Plugin\PluginHelper', 'getPlugins') ? Pluginhelper::getPlugins ( 'editors' ) : Pluginhelper::getPlugin ( 'editors' );

		foreach ( $aEditors as $sEditor ) {
			if (class_exists ( 'PlgEditor' . $sEditor->name, false )) {
				return true;
			}
		}

		return false;
	}

	/**
	 *
	 * @return boolean
	 */
	protected function pluginExclusions() {
		$user = $this->appInstance->getIdentity();

		$menuexcluded = $this->params->get ( 'menuexcluded', array () );
		$menuexcludedurl = $this->params->get ( 'menuexcludedurl', array () );
		
		// Check access levels intersection to ensure that users has access
		// Get users access levels based on user groups belonging
		$userAccessLevels = $user->getAuthorisedViewLevels();
		
		// Get chat access level from configuration, if set AKA param != array(0) go on with intersection
		$excludeAccess = false;
		$accessLevels = $this->params->get('pluginaccesslevels', array(0));
		if(is_array($accessLevels) && !in_array(0, $accessLevels, false)) {
			$intersectResult = array_intersect($userAccessLevels, $accessLevels);
			$excludeAccess = (bool)(count($intersectResult));
		}
		
		// Exclude by default the JSitemap bot for images/videos
		if (isset ( $_SERVER ['HTTP_USER_AGENT'] )) {
			$pattern = strtolower ( '/JSitemapbot/i' );
			if (preg_match ( $pattern, $_SERVER ['HTTP_USER_AGENT'] )) {
				$excludeAccess = true;
			}
		}
		// Exclude by default all other documents different than html
		$document = $this->appInstance->getDocument();
		if($document->getType() !== 'html') {
			$excludeAccess = true;
		}

		return (! $this->appInstance->isClient ( 'site' ) || $excludeAccess || ($this->appInstance->getInput()->get ( 'fastcachetaskexec', '', 'int' ) == 1) || ($this->appInstance->get ( 'offline', '0' ) && $user->guest) || $this->excludeEditorViews () || in_array ( $this->appInstance->getInput()->get ( 'Itemid', '', 'int' ), $menuexcluded ) || Fastcache\Helper::findExcludes ( $menuexcludedurl, FastcacheUri::getInstance ()->toString () ));
	}
	
	/**
	 * Get the sef string for the current language
	 *
	 * @access public
	 * @return string
	 */
	protected function getCurrentSefLanguage() {
		static $defaultLanguageSef;
		if($defaultLanguageSef) {
			return $defaultLanguageSef;
		}
		
		$knownLangs = LanguageHelper::getLanguages();
		
		// Setup predefined site language
		$defaultLanguageCode = $this->appInstance->getLanguage()->getTag();
		
		foreach ($knownLangs as $knownLang) {
			if($knownLang->lang_code == $defaultLanguageCode) {
				$defaultLanguageSef = $knownLang->sef;
				break;
			}
		}
		
		return $defaultLanguageSef;
	}

	/**
	 * PAGE CACHE METHODS
	 */
	
	/**
	 * Check if page cache should be saved for current request
	 *
	 * @return boolean
	 */
	protected function shouldSavePageCache() {
		if (!$this->params->get('htaccess_cache_enable', '0')) {
			return false;
		}
		
		if (!$this->appInstance->isClient('site')) {
			return false;
		}
		
		if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
			return false;
		}
		
		if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
			return false;
		}
		
		if ((isset($_SERVER['HTTP_ACCEPT']) && stripos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false) || (isset($_SERVER['CONTENT_TYPE']) && stripos($_SERVER['CONTENT_TYPE'], 'application/json') !== false)) {
			return false;
		}
		
		$user = $this->appInstance->getIdentity();
		if (!$user->guest) {
			return false;
		}
		
		if (!empty($_SERVER['QUERY_STRING'])) {
			return false;
		}
		
		$cache_exclude = $this->params->get('cache_exclude', '');
		if (!empty($cache_exclude)) {
			$excludeList = array_filter(array_map('trim', explode("\n", $cache_exclude)));
			$currentUri = FastcacheUri::getInstance()->toString();
			
			foreach ($excludeList as $excludePattern) {
				if (!empty($excludePattern) && stripos($currentUri, $excludePattern) !== false) {
					return false;
				}
			}
		}
		
		return true;
	}
	
	/**
	 * Get the cache file path for current page
	 *
	 * @return string
	 */
	protected function getPageCacheFilePath() {
		// Use REQUEST_URI directly (same as htaccess uses)
		$requestUri = $_SERVER['REQUEST_URI'];

		// Remove query string if present
		$requestUri = strtok($requestUri, '?');

		// Normalize trailing slash: strip it so /chi-siamo/ and /chi-siamo
		// always produce the same cache file. The trailing underscore is added
		// below by convention, independent of the original URL format.
		// This keeps the cache consistent regardless of the Joomla SEF
		// "Trailing slash for URLs" setting (No change / No slash / Slash).
		$requestUri = rtrim($requestUri, '/');

		// Replace slashes with underscores: /chi-siamo -> _chi-siamo
		$cacheId = str_replace('/', '_', $requestUri);

		// Handle empty (home page: "/" or "" after rtrim)
		if (empty($cacheId)) {
			$cacheId = '_';
		}

		// Ensure canonical format: always starts and ends with underscore
		if (substr($cacheId, 0, 1) !== '_') {
			$cacheId = '_' . $cacheId;
		}
		if (substr($cacheId, -1) !== '_') {
			$cacheId .= '_';
		}

		$cacheDir = JPATH_ROOT . '/cache/fastcache/page';

		if (!is_dir($cacheDir)) {
			mkdir($cacheDir, 0755, true);
			file_put_contents($cacheDir . '/index.html', '<html><body></body></html>');
		}

		return $cacheDir . '/' . $cacheId . '.html';
	}
	
	/**
	 * Save current page HTML to cache file
	 *
	 * @param string $html
	 * @return boolean
	 */
	protected function savePageCache($html) {
		try {
			$cacheFile = $this->getPageCacheFilePath();
			
			if (defined('JDEBUG') && JDEBUG) {
				$cacheDate = date('l, F d, Y h:i:s A');
				$html = str_replace('</body>', '<!-- Cached by FastCache on ' . $cacheDate . ' --> </body>', $html);
			}
			
			$result = file_put_contents($cacheFile, $html);
			
			return $result !== false;
		} catch (\Exception $e) {
			return false;
		}
	}
	
	/**
	 * Provide a hash for the default page cache plugin's key based on type of browser detected by Google font
	 *
	 * @param Event $event
	 * @return string $hash Calculated hash of browser type
	 */
	public function calculatePageCacheKey(Event $event) {
		// subparams: &$result
		$arguments = $event->getArguments();
		$result = isset($arguments['result']) ? $arguments['result'] : [];
		
		$browser = Fastcache\Browser::getInstance ();
		$hash = $browser->getFontHash ();
		
		$result[] = $hash;
		
		$event->setArgument('result', $result);

		return $result;
	}
	
	/**
	 * Provide the execution of special Fastcache tasks and the injection of lazy loading scripts
	 *
	 * @param Event $event
	 * @return void
	 */
	public function executeFastcacheGlobalTask(Event $event) {
		$jSpeedGlobalTask = $this->appInstance->getInput()->getCmd('fastcacheglobaltask', null);

		// Add clear cache button in admin only
		if ($this->params->get ( 'enable_clear_cache_button', 0 ) && $this->appInstance->isClient ( 'administrator' ) && $jSpeedGlobalTask == 'clearglobalcache' && $this->appInstance->getIdentity ()->authorise ( 'core.manage', 'com_plugins' )) {
			// Manage partial language translations
			$jLang = $this->appInstance->getLanguage ();
			$jLang->load ( 'plg_system_fastcache', JPATH_ADMINISTRATOR, 'en-GB', true, true );
			if ($jLang->getTag () != 'en-GB') {
				$jLang->load ( 'plg_system_fastcache', JPATH_ADMINISTRATOR, null, true, false );
			}

			$this->clearCache ();
		}

		// Previeni il Set-Cookie sulla POST di logout.
		// + espira joomla_user_state (e remember me) in modo esplicito.
		if ($this->appInstance->isClient('site')
				&& $this->params->get('enable_server_cache', 0)
				&& $this->appInstance->getInput()->getMethod() === 'POST'
				) {
			$task = $this->appInstance->getInput()->getCmd('task', '');
			
			if ($task === 'user.logout' || $task === 'user.menulogout') {
				
				// Impedisce a session_start()/session_regenerate_id() di settare cookie di sessione
				ini_set('session.use_cookies', '0');
				
				// Determina parametri cookie (path/domain) in modo coerente col sito
				$uri    = \Joomla\CMS\Uri\Uri::getInstance();
				$path   = rtrim($uri->base(true), '/');
				$path   = $path === '' ? '/' : $path . '/'; // Joomla spesso usa "/" o "/subdir/"
				$domain = $this->appInstance->get('cookie_domain', '') ?: ''; // se non configurato lascia vuoto
				
				$secure   = $this->appInstance->isHttpsForced();
				$httpOnly = true;
				
				// 1) Espira SEMPRE il cookie joomla_user_state (una tantum)
				if (isset($_COOKIE['joomla_user_state'])) {
					setcookie('joomla_user_state', '', [
							'expires'  => 1,
							'path'     => $path,
							'domain'   => $domain,
							'secure'   => $secure,
							'httponly' => $httpOnly,
							'samesite' => 'Lax',
					]);
				}
				
				// 2) (opzionale ma consigliato) espira il remember-me, se presente
				if (isset($_COOKIE['joomla_remember_me'])) {
					setcookie('joomla_remember_me', '', [
							'expires'  => 1,
							'path'     => $path,
							'domain'   => $domain,
							'secure'   => $secure,
							'httponly' => $httpOnly,
							'samesite' => 'Lax',
					]);
				}
				
				// 3) Se Joomla/user plugins aggiungono altri Set-Cookie durante logout/redirect,
				// li rimuoviamo. Ma ATTENZIONE: questa rimozione rimuoverebbe anche i Set-Cookie
				// di scadenza appena impostati. Quindi li reimpostiamo DOPO la remove.
				ob_start(function ($buffer) use ($path, $domain, $secure, $httpOnly) {
					if (!headers_sent()) {
						header_remove('Set-Cookie');
						
						// Re-espira i cookie critici dopo la remove
						setcookie('joomla_user_state', '', [
								'expires'  => 1,
								'path'     => $path,
								'domain'   => $domain,
								'secure'   => $secure,
								'httponly' => $httpOnly,
								'samesite' => 'Lax',
						]);
						
						setcookie('joomla_remember_me', '', [
								'expires'  => 1,
								'path'     => $path,
								'domain'   => $domain,
								'secure'   => $secure,
								'httponly' => $httpOnly,
								'samesite' => 'Lax',
						]);
					}
					return $buffer;
				});
			}
		}
	}
	
	/**
	 * Provide the execution of special Fastcache tasks and the injection of lazy loading scripts
	 *
	 * @param Event $event
	 * @return void
	 */
	public function executeFastcacheTask(Event $event) {
		// Add clear cache button in admin only
		if ($this->params->get ( 'enable_clear_cache_button', 0 ) && $this->appInstance->isClient ( 'administrator' ) && $this->appInstance->getIdentity ()->id && $this->appInstance->getIdentity ()->authorise ( 'core.manage', 'com_plugins' )) {
			$doc = $this->appInstance->getDocument ();
			$wa = $doc->getWebAssetManager ();
			$wa->registerAndUseScript ( 'fastcache.cachebutton', 'media/plg_fastcache/js/cachebutton.js' );
		}
		
		// Exclude by menu item
		$lazyLoadExcludeMenuitems = false;
		if (in_array ( $this->appInstance->getInput()->get ( 'Itemid', '', 'int' ), $this->params->get ( 'excludeLazyLoadMenuitem', array () ) )) {
			$lazyLoadExcludeMenuitems = true;
		}
		
		if ($this->params->get ( 'lazyload', '0' ) && 
			$this->params->get ( 'lazyload_mode', 'both' ) != 'native' && 
			! $this->pluginExclusions () && 
			!FastcacheHelper::findExcludes($this->params->get('excludeLazyLoadUrl', array()), FastcacheUri::getInstance()->toString()) && !$lazyLoadExcludeMenuitems) {
				
			$wa = $this->appInstance->getDocument()->getWebAssetManager();
			
			if(!$this->params->get ( 'lazyload_mode_speculative', 0 )) {
				$wa->registerAndUseScript ( 'fastcache.lazyload_loader', 'plg_fastcache/lazyload_loader.js' );
				
				if ($this->params->get ( 'lazyload_effects', '0' )) {
					$wa->registerAndUseStyle ( 'fastcache.lazyload_effects', 'plg_fastcache/lazyload_effects.css' );
					$wa->registerAndUseScript ( 'fastcache.lazyload_loader_effects', 'plg_fastcache/lazyload_loader_effects.js' );
				}
	
				// Lazyload autosize in JS mode
				if ($this->params->get ( 'lazyload_autosize', 0 ) == 2) {
					$wa->registerAndUseScript ( 'fastcache.lazyload_autosize', 'plg_fastcache/lazyload_autosize.js' );
				}
	
				$wa->registerAndUseScript ( 'fastcache.lazyload', 'plg_fastcache/lazyload.js' );
			} else {
				$wa->registerAndUseScript ( 'fastcache.speculative', 'plg_fastcache/speculative.js', [], ['defer'=>true]);
			}
		}
		
		// Check if the Instant Page feature is enabled, if so load the script
		if ($this->params->get ( 'enable_instant_page', '0' ) && ! $this->pluginExclusions ()) {
			$wa = $this->appInstance->getDocument()->getWebAssetManager();
			$wa->registerAndUseScript ( 'fastcache.instantpage', 'plg_fastcache/instantpage-5.2.0.js', [], ['defer'=>true]);
		}
		
		if($this->appInstance->isClient('site')) {
			return;
		}
		$matchTask = false;
		$jSpeedtask = $this->appInstance->getInput()->getCmd('fastcachetask', null);
		switch ($jSpeedtask) {
			case 'optimizehtaccess' :
				$htaccess = JPATH_ROOT . '/.htaccess';

				if (file_exists ( $htaccess )) {
					$contents = file_get_contents ( $htaccess );
					if (! preg_match ( '@\n?## START FASTCACHE OPTIMIZATIONS ##.*?## END FASTCACHE OPTIMIZATIONS ##@s', $contents )) {
						$sExpires = PHP_EOL;
						$sExpires .= '## START FASTCACHE OPTIMIZATIONS ##' . PHP_EOL;
						$sExpires .= '<IfModule mod_expires.c>' . PHP_EOL;
						$sExpires .= '  ExpiresActive on' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '# Default' . PHP_EOL;
						$sExpires .= '  ExpiresDefault "access plus 1 year"' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '# Application Cache' . PHP_EOL;
						$sExpires .= '  ExpiresByType text/cache-manifest "access plus 0 seconds"' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '# HTML Document' . PHP_EOL;
						$sExpires .= '  ExpiresByType text/html "access plus 0 seconds"' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '# Data documents' . PHP_EOL;
						$sExpires .= '  ExpiresByType text/xml "access plus 0 seconds"' . PHP_EOL;
						$sExpires .= '  ExpiresByType application/xml "access plus 0 seconds"' . PHP_EOL;
						$sExpires .= '  ExpiresByType application/json "access plus 0 seconds"' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '# Feed XML' . PHP_EOL;
						$sExpires .= '  ExpiresByType application/rss+xml "access plus 1 hour"' . PHP_EOL;
						$sExpires .= '  ExpiresByType application/atom+xml "access plus 1 hour"' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '# Favicon' . PHP_EOL;
						$sExpires .= '  ExpiresByType image/x-icon "access plus 1 week"' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '# Media: images, video, audio' . PHP_EOL;
						$sExpires .= '  ExpiresByType image/gif "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType image/png "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType image/jpg "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType image/jpeg "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType image/webp "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType video/ogg "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType audio/ogg "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType video/mp4 "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType video/webm "access plus 1 year"' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '# X-Component files' . PHP_EOL;
						$sExpires .= '  ExpiresByType text/x-component "access plus 1 year"' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '# Fonts' . PHP_EOL;
						$sExpires .= '  ExpiresByType application/font-ttf "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType font/opentype "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType application/font-woff "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType application/font-woff2 "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType image/svg+xml "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType application/vnd.ms-fontobject "access plus 1 year"' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '# CSS and JavaScript' . PHP_EOL;
						$sExpires .= '  ExpiresByType text/css "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType text/javascript "access plus 1 year"' . PHP_EOL;
						$sExpires .= '  ExpiresByType application/javascript "access plus 1 year"' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '  <IfModule mod_headers.c>' . PHP_EOL;
						$sExpires .= '    Header append Cache-Control "public"' . PHP_EOL;
						$sExpires .= '    <FilesMatch ".(js|css|xml|gz|html)$">' . PHP_EOL;
						$sExpires .= '       Header append Vary: Accept-Encoding' . PHP_EOL;
						$sExpires .= '    </FilesMatch>' . PHP_EOL;
						$sExpires .= '  </IfModule>' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '</IfModule>' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '<IfModule mod_deflate.c>' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE text/html' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE text/css' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE text/javascript' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE text/xml' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE text/plain' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE image/x-icon' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE image/svg+xml' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/rss+xml' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/javascript' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/x-javascript' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/xml' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/xhtml+xml' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/font' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/font-truetype' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/font-ttf' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/font-otf' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/font-opentype' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/font-woff' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/font-woff2' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE application/vnd.ms-fontobject' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE font/ttf' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE font/otf' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE font/opentype' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE font/woff' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType DEFLATE font/woff2' . PHP_EOL;
						$sExpires .= '# GZip Compression' . PHP_EOL;
						$sExpires .= 'BrowserMatch ^Mozilla/4 gzip-only-text/html' . PHP_EOL;
						$sExpires .= 'BrowserMatch ^Mozilla/4\.0[678] no-gzip' . PHP_EOL;
						$sExpires .= 'BrowserMatch \bMSIE !no-gzip !gzip-only-text/html' . PHP_EOL;
						$sExpires .= '</IfModule>' . PHP_EOL;
						$sExpires .= '' . PHP_EOL;
						$sExpires .= '<IfModule mod_brotli.c>' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS text/html' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS text/css' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS text/javascript' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS text/xml' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS text/plain' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS image/x-icon' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS image/svg+xml' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/rss+xml' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/javascript' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/x-javascript' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/xml' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/xhtml+xml' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/font' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/font-truetype' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/font-ttf' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/font-otf' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/font-opentype' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/font-woff' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/font-woff2' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS application/vnd.ms-fontobject' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS font/ttf' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS font/otf' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS font/opentype' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS font/woff' . PHP_EOL;
						$sExpires .= 'AddOutputFilterByType BROTLI_COMPRESS font/woff2' . PHP_EOL;
						$sExpires .= '</IfModule>' . PHP_EOL;
						$sExpires .= '## END FASTCACHE OPTIMIZATIONS ##' . PHP_EOL;

						$writtenFile = file_put_contents ( $htaccess, $sExpires, FILE_APPEND );
						if($writtenFile) {
							$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_HTACCESS_SUCCESSFULLY_CONFIGURED'));
						}
					} else {
						$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_HTACCESS_ALREADY_CONFIGURED'));
					}
				} else {
					$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_HTACCESS_MISSING'));
				}
				
				$matchTask = true;
			break;
			
			case 'writePageCacheRules':
				$htaccess = JPATH_ROOT . '/.htaccess';

				if (file_exists($htaccess)) {
					$contents = file_get_contents($htaccess);

					// Get Joomla session cookie name
					$sessionName = Factory::getApplication()->get('session_name', 'joomla_session');

					// Level-based matching compatible with all Apache versions
					{
						$sPageCache = PHP_EOL . '## BEGIN HTACCESS PAGE CACHING - FASTCACHE ##' . PHP_EOL;
						$sPageCache .= '## 1.1' . PHP_EOL;
						$sPageCache .= 'RewriteEngine On' . PHP_EOL;
						$sPageCache .= 'RewriteCond %{REQUEST_URI} !^/cache/fastcache/page/ [NC]' . PHP_EOL;
						$sPageCache .= 'RewriteRule ^ - [E=FASTCACHE_LEVEL:MISS]' . PHP_EOL;

						// HOME
						$sPageCache .= '# ===== HOME =====' . PHP_EOL;
						$sPageCache .= 'RewriteCond %{REQUEST_METHOD} ^GET$ [NC]' . PHP_EOL;
						$sPageCache .= 'RewriteCond %{HTTP_X_REQUESTED_WITH} !^XMLHttpRequest$ [NC]' . PHP_EOL;
						$sPageCache .= 'RewriteCond %{QUERY_STRING} ^$' . PHP_EOL;
						$sPageCache .= 'RewriteCond %{HTTP_COOKIE} !(' . $sessionName . ') [NC]' . PHP_EOL;
						$sPageCache .= 'RewriteCond %{REQUEST_URI} !^/(administrator|api) [NC]' . PHP_EOL;
						$sPageCache .= 'RewriteCond %{REQUEST_URI} ^/$' . PHP_EOL;
						$sPageCache .= 'RewriteCond %{DOCUMENT_ROOT}/cache/fastcache/page/_.html -f' . PHP_EOL;
						$sPageCache .= 'RewriteRule ^$ /cache/fastcache/page/_.html [L,E=FASTCACHE_LEVEL:HITHOME]' . PHP_EOL;
						$sPageCache .= PHP_EOL;

						// Generate rules for URL levels 1-10
						// Trailing slash is optional (/?) so both /chi-siamo and /chi-siamo/
						// match the same cached file, regardless of Joomla SEF trailing slash setting
						for ($level = 1; $level <= 10; $level++) {
							$sPageCache .= '# ===== LEVEL ' . $level . ' =====' . PHP_EOL;
							$sPageCache .= 'RewriteCond %{REQUEST_METHOD} ^GET$ [NC]' . PHP_EOL;
							$sPageCache .= 'RewriteCond %{HTTP_X_REQUESTED_WITH} !^XMLHttpRequest$ [NC]' . PHP_EOL;
							$sPageCache .= 'RewriteCond %{QUERY_STRING} ^$' . PHP_EOL;
							$sPageCache .= 'RewriteCond %{HTTP_COOKIE} !(' . $sessionName . ') [NC]' . PHP_EOL;
							$sPageCache .= 'RewriteCond %{REQUEST_URI} !^/(administrator|api) [NC]' . PHP_EOL;

							// URI pattern: trailing slash optional with /?$
							$sPageCache .= 'RewriteCond %{REQUEST_URI} ^/';
							for ($i = 1; $i <= $level; $i++) {
								$sPageCache .= '([^/]+)';
								if ($i < $level) {
									$sPageCache .= '/';
								}
							}
							$sPageCache .= '/?$' . PHP_EOL;

							// File existence check - always maps to _segment1_segment2_.html
							$sPageCache .= 'RewriteCond %{DOCUMENT_ROOT}/cache/fastcache/page/_';
							for ($i = 1; $i <= $level; $i++) {
								$sPageCache .= '%' . $i;
								if ($i < $level) {
									$sPageCache .= '_';
								}
							}
							$sPageCache .= '_.html -f' . PHP_EOL;

							// Serve cached file
							$sPageCache .= 'RewriteRule ^ /cache/fastcache/page/_';
							for ($i = 1; $i <= $level; $i++) {
								$sPageCache .= '%' . $i;
								if ($i < $level) {
									$sPageCache .= '_';
								}
							}
							$sPageCache .= '_.html [L,E=FASTCACHE_LEVEL:HITL' . $level . ']' . PHP_EOL;
							$sPageCache .= PHP_EOL;
						}

						// Debug header
						$sPageCache .= '# ===== DEBUG HEADER (set only when FASTCACHE_LEVEL is present) =====' . PHP_EOL;
						$sPageCache .= 'Header always set X-HST-FASTCACHE-FS "%{FASTCACHE_LEVEL}e"' . PHP_EOL;
						$sPageCache .= PHP_EOL;
						$sPageCache .= '## END HTACCESS PAGE CACHING - FASTCACHE ##' . PHP_EOL;
					}

					// Regex robusta che matcha TUTTE le occorrenze (anche duplicate, anche legacy con "2.4")
					$regex = '~^[ \t]*## BEGIN HTACCESS PAGE CACHING - FASTCACHE[^\r\n]*\R.*?^[ \t]*## END HTACCESS PAGE CACHING - FASTCACHE[^\r\n]*\R?~sm';

					// 1) Rimuovi TUTTE le occorrenze (anche duplicate, anche legacy)
					$clean = preg_replace($regex, '', $contents, -1, $count);

					// 2) Se c'erano blocchi esistenti, riscrivi inserendo UN SOLO blocco canonico
					if ($count > 0) {
						$writtenFile = file_put_contents($htaccess, $sPageCache . $clean);
						if ($writtenFile) {
							$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_HTACCESS_PAGE_CACHE_ADDED'));
						}
					} else {
						// 3) Il blocco non c'era: aggiungilo (prepend)
						$writtenFile = file_put_contents($htaccess, $sPageCache . $contents);
						if ($writtenFile) {
							$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_HTACCESS_PAGE_CACHE_ADDED'));
						}
					}
				} else {
					$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_HTACCESS_MISSING'));
				}

				$matchTask = true;
			break;

			case 'removePageCacheRules':
				$htaccess = JPATH_ROOT . '/.htaccess';

				if (file_exists($htaccess)) {
					$contents = file_get_contents($htaccess);
					// Regex robusta che matcha TUTTE le occorrenze (anche duplicate, anche legacy con "2.4")
					$regex = '~^[ \t]*## BEGIN HTACCESS PAGE CACHING - FASTCACHE[^\r\n]*\R.*?^[ \t]*## END HTACCESS PAGE CACHING - FASTCACHE[^\r\n]*\R?~sm';

					$clean_contents = preg_replace($regex, '', $contents, -1, $count);

					if ($count > 0) {
						$writtenFile = file_put_contents($htaccess, $clean_contents);
						if ($writtenFile) {
							$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_HTACCESS_PAGE_CACHE_REMOVED'));
						}
					} else {
						$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_HTACCESS_PAGE_CACHE_ALREADY_REMOVED'));
					}
				} else {
					$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_HTACCESS_MISSING'));
				}

				$matchTask = true;
			break;
				
			case 'restorehtaccess':
				$htaccess = JPATH_ROOT . '/.htaccess';
				if (file_exists ( $htaccess )) {
					$contents = file_get_contents ( $htaccess );
					$regex = '@\n?## START FASTCACHE OPTIMIZATIONS ##.*?## END FASTCACHE OPTIMIZATIONS ##@s';
					
					$clean_contents = preg_replace ( $regex, '', $contents, - 1, $count );
					
					if ($count > 0) {
						$writtenFile = file_put_contents ( $htaccess, $clean_contents );
						if($writtenFile) {
							$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_HTACCESS_SUCCESSFULLY_RESTORED'));
						}
					} else {
						$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_HTACCESS_ALREADY_RESTORED'));
					}
				} else {
					$this->appInstance->enqueueMessage(Text::_('PLG_FASTCACHE_HTACCESS_MISSING'));
				}
				
				$matchTask = true;
			break;
			
			case 'clearcache' :
				$this->clearCache();
				
				$matchTask = true;
			break;
		}
		if ($matchTask) {
			// Check if JSON response is requested
			$format = $this->appInstance->getInput()->get('fastcacheformat', '');
			
			if ($format === 'json') {
				// Return JSON response
				$messages = $this->appInstance->getMessageQueue();
				
				// Prepare response
				$response = array(
						'success' => true,
						'messages' => array()
				);
				
				// Extract messages
				foreach ($messages as $message) {
					$response['messages'][] = array(
							'type' => $message['type'],
							'message' => $message['message']
					);
				}
				
				// Clear message queue to prevent showing them again
				$this->appInstance->getSession()->set('application.queue', null);
				
				// Set JSON header and output
				header('Content-Type: application/json');
				echo json_encode($response);
				
				// Close application
				$this->appInstance->close();
			} else {
				// Standard redirect behavior
				$oUri = clone Uri::getInstance();
				$oUri->delVar('fastcachetask');
				$this->appInstance->redirect($oUri->toString());
			}
		}
	}
	
	
	/**
	 * Handler for onAfterRender - runs optimizations first, then saves page cache independently
	 *
	 * @param Event $event
	 * @return void
	 */
	public function onAfterRenderHandler(Event $event) {
		$this->executeOptimizations($event);
		$this->savePageCacheOnRender($event);
	}

	/**
	 * Main plugin execution method, here happens the magic and optimizations of the output buffer
	 *
	 * @param Event $event
	 * @return boolean
	 * @throws Exception
	 */
	public function executeOptimizations(Event $event) {
		if ($this->pluginExclusions ()) {
			return false;
		}
		
		$sHtml = $this->appInstance->getBody ();
		
		if (! Fastcache\Helper::validateHtml ( $sHtml )) {
			return false;
		}
		
		if ($this->appInstance->getInput()->get ( 'fastcachetaskexec' ) == '2') {
			echo $sHtml;
			while ( @ob_end_flush () )
				;
			exit ();
		}
		
		try {
			FastcacheAutoLoader ( 'Fastcache\Optimizer' );
			
			$sOptimizedHtml = Fastcache\Optimizer::optimize ( $this->params, $sHtml );
		} catch ( \Exception $e ) {
			$sOptimizedHtml = $sHtml;
		}
		
		if($this->params->get ( 'html_minify_level', 0 ) == 2) {
			$sOptimizedHtml = preg_replace('/.css \/>/i', '.css data-css=""/>',  $sOptimizedHtml);
		}
		
		if ($this->params->get ( 'enable_instant_page', '0' ) && $this->params->get ( 'instant_page_delay', 'fast' ) == 'slow' && ! $this->pluginExclusions ()) {
			$sOptimizedHtml = preg_replace('/<body/i', '<body data-instant-intensity="150"',  $sOptimizedHtml);
		}
		
		$this->appInstance->setBody ( $sOptimizedHtml );
	}

	/**
	 * Independent page cache handler - saves the final HTML to static file
	 * This runs independently from optimizations, so page cache works even
	 * when no other optimization is active.
	 *
	 * @param Event $event
	 * @return void
	 */
	public function savePageCacheOnRender(Event $event) {
		if ($this->shouldSavePageCache()) {
			$sHtml = $this->appInstance->getBody();

			if (!empty($sHtml)) {
				$this->savePageCache($sHtml);
			}
		}
	}

	/**
	 * Gestisce la cache server-side (Varnish/CDN) su OGNI richiesta frontend
	 * Invia header HTTP custom e killa cookie di sessione per utenti non loggati
	 * Questo metodo viene eseguito automaticamente su ogni pagina del sito frontend
	 *
	 * @param Event $event
	 * @return void
	 */
	public function addCacheHeaders(Event $event) {
		// Solo frontend - non eseguire in backend
		if (!$this->appInstance->isClient('site')) {
			return;
		}
		
		// Parametri plugin per cache server-side
		$enableServerCache = (int) $this->params->get('enable_server_cache', 0);
		
		// Se cache server-side è disabilitata, non inviare NESSUN header - esci subito
		if ($enableServerCache != 1) {
			return;
		}
		
		// Escludi tutto ciò che non è HTML (xml/json/feed/pdf/raw ecc.)
		$document = $this->appInstance->getDocument();
		if (!$document || $document->getType() !== 'html') {
			return;
		}
		
		$ttl = (int) $this->params->get('server_cache_ttl', 3600);
		$excludedUrls = $this->params->get('server_cache_excluded_urls', '');

		// Verifica se URL corrente è escluso dalla cache
		$excluded = $this->isServerCacheUrlExcluded($excludedUrls);

		// Gestione header cache HTTP - SOLO se cache è abilitata
		if ($excluded > 0) {
			// URL escluso dalla cache - invia header NO cache
			$this->appInstance->setHeader('X-HST-CACHE-Enabled', 'NO:Url Exclusion Matched', true);
			$this->appInstance->setHeader('X-HST-CACHE-ttl', '0', true);
			$this->appInstance->setHeader('X-Logged-In', 'False', true);
		} else {
			$user = $this->appInstance->getIdentity();
			
			if (!$user->guest) {
				// Utente loggato - NO cache
				$this->appInstance->setHeader('X-HST-CACHE-Enabled', 'false:User is logged in', true);
				$this->appInstance->setHeader('X-HST-CACHE-ttl', '0', true);
				$this->appInstance->setHeader('X-Logged-In', 'True', true);
			} else {
				// Utente NON loggato - SI cache
				$this->appInstance->setHeader('X-HST-CACHE-Enabled', 'true', true);
				$this->appInstance->setHeader('X-HST-CACHE-ttl', (string) $ttl, true);
				$this->appInstance->setHeader('X-Logged-In', 'False', true);
			}
		}
	}

	/**
	 * Event to manipulate the menu item dashboard in backend
	 *
	 * @param Event $event
	 * @subparam   array  &$policy  The privacy policy status data, passed by reference, with keys "published" and "editLink"
	 *
	 * @return  void
	 */
	public function processMenuItemsDashboard(Event $event) {
		$arguments = $event->getArguments();
		
		// Kill com_joomlaupdate informations about extensions missing updater info, leave only main one
		$document = $this->appInstance->getDocument();
		if(!$this->appInstance->get('jextstore_joomlaupdate_script') && $this->appInstance->getInput()->get('option') == 'com_joomlaupdate' && !$this->appInstance->getInput()->get('view') && !$this->appInstance->getInput()->get('task')) {
			$document->getWebAssetManager()->addInlineScript ("
				window.addEventListener('DOMContentLoaded', function(e) {
					if(document.querySelector('#preupdatecheck')) {
						var jextensionsIntervalCount = 0;
						var jextensionsIntervalTimer = setInterval(function() {
						    [].slice.call(document.querySelectorAll('#compatibilityTable1 tbody tr th.exname')).forEach(function(th) {
						        let txt = th.innerText;
						        if (txt && txt.toLowerCase().match(/jsitemap|gdpr|gptranslate|jpagebuilder|responsivizer|jchatsocial|jcomment|jshortcodes|jrealtime|fastcache|jredirects|vsutility|visualstyles|visual\sstyles|instant\sfacebook\slogin|instantpaypal|screen\sreader|fastcache|jamp/i)) {
						            th.parentElement.style.display = 'none';
						            th.parentElement.classList.remove('error');
									th.parentElement.classList.add('jextcompatible');
						        }
						    });
							[].slice.call(document.querySelectorAll('#compatibilityTable2 tbody tr th.exname')).forEach(function(th) {
						        let txt = th.innerText;
						        if (txt && txt.toLowerCase().match(/jsitemap|gdpr|gptranslate|jpagebuilder|responsivizer|jchatsocial|jcomment|jshortcodes|jrealtime|fastcache|jredirects|vsutility|visualstyles|visual\sstyles|instant\sfacebook\slogin|instantpaypal|screen\sreader|fastcache|jamp/i)) {
									th.parentElement.classList.remove('error');
									th.parentElement.classList.add('jextcompatible');
						            let smallDiv = th.querySelector(':scope div.small');
									if(smallDiv) {
										smallDiv.style.display = 'none';
									}
						        }
						    });
							if (document.querySelectorAll('#compatibilityTable0 tbody tr').length == 0 &&
								document.querySelectorAll('#compatibilityTable1 tbody tr:not(.jextcompatible)').length == 0 &&
								document.querySelectorAll('#compatibilityTable2 tbody tr:not(.jextcompatible)').length == 0) {
						        [].slice.call(document.querySelectorAll('#preupdatecheckbox, #preupdateCheckCompleteProblems')).forEach(function(element) {
						            element.style.display = 'none';
						        });
								if(document.querySelector('#noncoreplugins')) {
									document.querySelector('#noncoreplugins').checked = true;
								}
								if(document.querySelector('button.submitupdate')) {
							        document.querySelector('button.submitupdate').disabled = false;
							        document.querySelector('button.submitupdate').classList.remove('disabled');
								}
								if(document.querySelector('#joomlaupdate-precheck-extensions-tab span.fa')) {
									let tabIcon = document.querySelector('#joomlaupdate-precheck-extensions-tab span.fa');
									tabIcon.classList.remove('fa-times');
									tabIcon.classList.remove('text-danger');
									tabIcon.classList.remove('fa-exclamation-triangle');
									tabIcon.classList.remove('text-warning');
									tabIcon.classList.add('fa-check');
									tabIcon.classList.add('text-success');
								}
						    };
					
							if (document.querySelectorAll('#compatibilityTable0 tbody tr').length == 0) {
								if(document.querySelectorAll('#compatibilityTable1 tbody tr:not(.jextcompatible)').length == 0) {
									let compatibilityTable1 = document.querySelector('#compatibilityTable1');
									if(compatibilityTable1) {
										compatibilityTable1.style.display = 'none';
									}
								}
								clearInterval(jextensionsIntervalTimer);
							}
					
						    jextensionsIntervalCount++;
						}, 1000);
					};
				});");
			$this->appInstance->set('jextstore_joomlaupdate_script', true);
		}
	}
	
	/**
	 * Event to query the Google PageSpeed Insights API
	 *
	 * @param Event $event
	 *
	 * @return  void
	 */
	public function processAjaxStuffsPagespeedCache(Event $event) {
		// Handle token request per il login session bridge
		$task = $this->appInstance->getInput()->getCmd('task', '');
		if ($task === 'token') {
			$purgeDebug = [];
			
			if ($this->params->get('enable_server_cache', 0)) {
				$purgeToken = trim($this->params->get('server_cache_token', ''));
				
				if (!empty($purgeToken)) {
					$host = rtrim(\Joomla\CMS\Uri\Uri::root(false), '/');
					
					try {
						// Ricevi URL pagina corrente dal JS
						$purgeUrl = $this->appInstance->getInput()->getString('purge_url', '');
						
						// Estrai path come WP
						$path = '/';
						if (!empty($purgeUrl)) {
							$parsed = parse_url($purgeUrl);
							if (!empty($parsed['path'])) {
								$path = $parsed['path'];
							}
						}
						
						// Normalizza come WP: se "/" => homepage
						if ($path === '/' || $path === '') {
							$path = '/'; // endpoint base
						}
						
						// PURGE SOLO questa pagina: exact
						$ok = $this->execServerCachePurge($host, $purgeToken, 'exact', $path);
						
						$purgeDebug['success'] = (bool) $ok;
						$purgeDebug['mode']    = 'exact';
						$purgeDebug['path']    = $path;
					} catch (\Throwable $e) {
						$purgeDebug['success'] = false;
						$purgeDebug['error']   = $e->getMessage();
					}
				} else {
					$purgeDebug['success'] = false;
					$purgeDebug['error']   = 'Missing server_cache_token';
				}
			}

			$token = \Joomla\CMS\Session\Session::getFormToken();
			$event->setArgument('result', [['token' => $token, 'purgeDebug' => $purgeDebug]]);
			return;
		}

		// Ensure that at least a language is available for the backend and locale sent to Google otherwise the API fails
		$language = $this->getCurrentSefLanguage();
		$locale = $language ? $language : 'en';
		$response = new \stdClass();
		
		// Build the purified domain to scrape using the host only
		$linkUrl = $this->params->get('pagespeedtest_domain_url', Uri::root(false));
		$hostDomain = rawurlencode ( $linkUrl );
		$customApiKey = trim($this->params->get ( 'google_pagespeed_api_key', ''));
		$apiKey = $customApiKey ? $customApiKey : 'AIzaSyDBN6utYmIBNQ2IVlLcY7-42S9GuKHBIfQ'; 

		$urlDesktop = "https://content.googleapis.com/pagespeedonline/v5/runPagespeed?url=$hostDomain&key=$apiKey&strategy=desktop&category=performance&locale=" . $locale;
		$urlMobile = "https://content.googleapis.com/pagespeedonline/v5/runPagespeed?url=$hostDomain&key=$apiKey&strategy=mobile&category=performance&locale=" . $locale;
		
		try {
			// Fetch remote data to scrape
			$connectionAdapter = \Fastcache\FileScanner::getInstance ();
			$httpResponseDesktop = $connectionAdapter->getFileContents ( $urlDesktop, null, array (), '', 60);
			$httpResponseMobile = $connectionAdapter->getFileContents ( $urlMobile, null, array (), '', 60);
			
			// Check if HTTP status code is 200 OK
			if ($httpResponseDesktop) {
				$decodedApiResponse = json_decode($httpResponseDesktop, true);
				if(is_array($decodedApiResponse)) {
					// Calculate the score category, range, colors for labels and sliders
					$response->pagespeedDesktop = isset($decodedApiResponse['lighthouseResult']['categories']['performance']) ? (int)($decodedApiResponse['lighthouseResult']['categories']['performance']['score'] * 100) : -1;
					
					$response->fcpDesktop = number_format($decodedApiResponse['lighthouseResult']['audits']['first-contentful-paint']['numericValue'] / 1000, 1);
					$response->siDesktop = number_format($decodedApiResponse['lighthouseResult']['audits']['speed-index']['numericValue'] / 1000, 1);
					$response->lcpDesktop = number_format($decodedApiResponse['lighthouseResult']['audits']['largest-contentful-paint']['numericValue'] / 1000, 1);
					
					$response->interactiveDesktop = number_format($decodedApiResponse['lighthouseResult']['audits']['interactive']['numericValue'] / 1000, 1);
					$response->tbtDesktop = intval($decodedApiResponse['lighthouseResult']['audits']['total-blocking-time']['numericValue']);
					$response->clsDesktop = number_format($decodedApiResponse['lighthouseResult']['audits']['cumulative-layout-shift']['numericValue'], 3);
					
					$response->screenShotDesktop = $decodedApiResponse['lighthouseResult']['audits']['final-screenshot']['details']['data'];
				}
			}
			// Check if HTTP status code is 200 OK
			if ($httpResponseMobile) {
				$decodedApiResponse = json_decode($httpResponseMobile, true);
				if(is_array($decodedApiResponse)) {
					// Calculate the score category, range, colors for labels and sliders
					$response->pagespeedMobile = isset($decodedApiResponse['lighthouseResult']['categories']['performance']) ? (int)($decodedApiResponse['lighthouseResult']['categories']['performance']['score'] * 100) : -1;
					
					$response->fcpMobile = number_format($decodedApiResponse['lighthouseResult']['audits']['first-contentful-paint']['numericValue'] / 1000, 1);
					$response->siMobile = number_format($decodedApiResponse['lighthouseResult']['audits']['speed-index']['numericValue'] / 1000, 1);
					$response->lcpMobile = number_format($decodedApiResponse['lighthouseResult']['audits']['largest-contentful-paint']['numericValue'] / 1000, 1);
					
					$response->interactiveMobile = number_format($decodedApiResponse['lighthouseResult']['audits']['interactive']['numericValue'] / 1000, 1);
					$response->tbtMobile = intval($decodedApiResponse['lighthouseResult']['audits']['total-blocking-time']['numericValue']);
					$response->clsMobile = number_format($decodedApiResponse['lighthouseResult']['audits']['cumulative-layout-shift']['numericValue'], 3);
					
					$response->screenShotMobile = $decodedApiResponse['lighthouseResult']['audits']['final-screenshot']['details']['data'];
				}
			}
			
			$response->analyzedUrl = $linkUrl;
		} catch ( \Exception $e ) {
			// Go on with the next API without blocking exception
		}
		
		$event->setArgument('result', $response);
	}
	
	/**
	 * Handler unico per onBeforeCompileHead - dispatcha ai sotto-metodi
	 *
	 * @param Event $event
	 * @return void
	 */
	public function onBeforeCompileHeadHandler(Event $event) {
		$this->killSessionCookie($event);
		$this->injectLoginSessionBridge($event);
	}

	/**
	 * Ultimo evento prima dell'invio della response - SEMPRE eseguito
	 * Qui manipoliamo direttamente gli header HTTP per killare il cookie di sessione
	 *
	 * Strategia per Varnish: per utenti guest vogliamo ZERO Set-Cookie header nella response
	 * - Guest senza cookie in arrivo: solo header_remove, nessun cookie va e nessun cookie torna
	 * - Guest con cookie residuo: header_remove + setcookie expired (una tantum per pulirlo)
	 * - Utente loggato: non toccare nulla
	 * - Request POST: non toccare (login, form contatti, registrazione... necessitano della sessione
	 *   per il token CSRF. Le POST non vengono cachate da Varnish quindi è irrilevante)
	 *
	 * Per i form di login (modulo presente su tutte le pagine): un JS intercetta il submit,
	 * chiama un endpoint AJAX per stabilire la sessione e ottenere un token CSRF fresco,
	 * aggiorna il form e poi fa il submit reale. Vedi injectLoginSessionBridge().
	 *
	 * @param Event $event
	 * @return void
	 */
	public function killSessionCookie(Event $event) {
		// Solo frontend
		if (!$this->appInstance->isClient('site')) {
			return;
		}

		// Solo se cache server-side è abilitata
		if (!$this->params->get('enable_server_cache', 0)) {
			return;
		}

		// Non killare il cookie sulle POST: servono la sessione per il token CSRF
		// (login, registrazione, form contatti, ecc.) e Varnish non cacha le POST comunque.
		// NB: il logout POST è gestito in executeFastcacheGlobalTask() via ini_set('session.use_cookies', 0).
		if ($this->appInstance->getInput()->getMethod() === 'POST') {
			return;
		}

		// Non killare il cookie sulle request AJAX (format=json/raw) e com_ajax:
		// sono usate dal login session bridge per stabilire la sessione e ottenere il token.
		// Varnish non cacha request con query string com_ajax comunque.
		$format = $this->appInstance->getInput()->getCmd('format', 'html');
		$option = $this->appInstance->getInput()->getCmd('option', '');
		if ($format !== 'html' || $option === 'com_ajax') {
			return;
		}

		// Solo per utenti guest
		$user = $this->appInstance->getIdentity();
		if (!$user->guest) {
			return;
		}

		if (!headers_sent()) {
			$sessionName = $this->appInstance->getSession()->getName();

			// Controlla se il browser ha inviato un cookie di sessione nella request
			$hasSessionCookie = isset($_COOKIE[$sessionName]);

			// Rimuovi TUTTI i Set-Cookie header generati da session_start()
			header_remove('Set-Cookie');

			if ($hasSessionCookie) {
				// Il browser ha un cookie residuo da una sessione precedente:
				// dobbiamo mandare un Set-Cookie expired per farglielo eliminare.
				// Dalla prossima visita non avrà più il cookie e cadrà nel caso sotto.
				setcookie(
						$sessionName,
						'',
						[
								'expires' => 1,
								'path' => '/',
								'domain' => '',
								'secure' => $this->appInstance->isHttpsForced(),
								'httponly' => true,
								'samesite' => 'Lax'
						]
				);
			}
			// Se NON ha cookie in arrivo: non facciamo nulla dopo header_remove.
			// Risultato: ZERO header Set-Cookie nella response → Varnish può cachare.

			// Abilita la cachability dell'applicazione Joomla. Senza questo,
			// respond() in AbstractWebApplication setta Cache-Control: no-store,
			// Pragma: no-cache, Expires nel passato — tutti header che impediscono
			// a Varnish di cachare. Con allowCache(true), Joomla non aggiunge
			// quegli header e possiamo controllare noi il Cache-Control.
			$this->appInstance->allowCache(true);

			// Setta l'header Cache-Control per Varnish con il TTL configurato
			$ttl = (int) $this->params->get('server_cache_ttl', 3600);
			$this->appInstance->setHeader('Cache-Control', 'public, s-maxage=' . $ttl, true);
		}
	}

	/**
	 * Inietta il JavaScript bridge per il login su pagine cachate senza sessione.
	 *
	 * Problema: le pagine servite da Varnish non hanno cookie di sessione, quindi il
	 * token CSRF nel form di login è "congelato" e non validabile.
	 *
	 * Soluzione: al submit del form di login, il JS:
	 * 1. Blocca il submit nativo
	 * 2. Chiama l'endpoint AJAX che stabilisce una sessione (restituisce cookie + token)
	 * 3. Sostituisce il campo hidden del token nel form con quello fresco
	 * 4. Fa il submit reale del form
	 *
	 * @param Event $event  onBeforeCompileHead event
	 * @return void
	 */
	public function injectLoginSessionBridge(Event $event) {
		// Solo frontend
		if (!$this->appInstance->isClient('site')) {
			return;
		}

		// Solo se cache server-side è abilitata
		if (!$this->params->get('enable_server_cache', 0)) {
			return;
		}

		// Solo per utenti guest (loggati hanno già la sessione)
		$user = $this->appInstance->getIdentity();
		if (!$user->guest) {
			return;
		}

		// Solo su GET (le POST non vengono cachate e hanno già la sessione)
		if ($this->appInstance->getInput()->getMethod() !== 'GET') {
			return;
		}

		$tokenEndpoint = \Joomla\CMS\Uri\Uri::base() . 'index.php?option=com_ajax&plugin=fastcache&group=system&format=json&task=token';

		$js = <<<JS
document.addEventListener('DOMContentLoaded',function(){
	document.querySelectorAll('form[action] input[type="hidden"][name="task"][value="user.login"]').forEach(function(taskInput){
		var form=taskInput.closest('form');
		if(!form)return;
		form.addEventListener('submit',function(e){
			if(form.dataset.fcReady)return;
			e.preventDefault();
			var btn=form.querySelector('button[type="submit"]');
			if(btn)btn.disabled=true;
			var purgeUrl=encodeURIComponent(window.location.href);
			fetch('{$tokenEndpoint}&purge_url='+purgeUrl,{method:'POST',credentials:'same-origin'})
			.then(function(r){return r.json()})
			.then(function(data){
				var token=data.data&&data.data[0]?data.data[0].token:null;
				if(!token){if(btn)btn.disabled=false;form.submit();return}
				var inputs=form.querySelectorAll('input[type="hidden"]');
				for(var i=0;i<inputs.length;i++){
					if(inputs[i].name!=='option'&&inputs[i].name!=='task'&&inputs[i].name!=='return'&&inputs[i].value==='1'&&inputs[i].name.length===32){
						inputs[i].name=token;
						break;
					}
				}
				form.dataset.fcReady='1';
				form.submit();
			})
			.catch(function(){
				if(btn)btn.disabled=false;
				form.dataset.fcReady='1';
				form.submit();
			});
		});
	});
});
JS;

		$this->appInstance->getDocument()->addScriptDeclaration($js);
	}

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return  array
	 *
	 * @since 4.0.0
	 */
	public static function getSubscribedEvents(): array {
		return [
				'onPageCacheGetKey' => 'calculatePageCacheKey',
				'onAfterInitialise' => 'executeFastcacheGlobalTask',
				'onAfterRoute' => 'addCacheHeaders',
				'onAfterDispatch' => 'executeFastcacheTask',
				'onBeforeCompileHead' => 'onBeforeCompileHeadHandler',
				'onAfterRender' => 'onAfterRenderHandler',
				'onPreprocessMenuItems' => 'processMenuItemsDashboard',
				'onAjaxFastcache' => 'processAjaxStuffsPagespeedCache'
		];
	}
	
	/**
	 * Plugin constructor
	 *
	 * @access public
	 */
	public function __construct($subject, $config = []) {
		parent::__construct ( $subject, $config );
		
		// Init application
		$this->appInstance = Factory::getApplication();
	}
}
